const express = require('express');
const router = express.Router();
const { uploadMiddleware } = require('../config/uploads');
const {
    getUser,
    getUserById,
    updateUser,
    deleteUser,
    changePassword,
    addVendor,
    addClient
} = require('../src/controllers/user');

const {
    uploadDocument,
    getDocumentById,
    deleteDocumentById,
    deleteSelectedDocumentByIds,
    uploadZipDocument,
    getZipExtractedDocuments
} = require('../src/controllers/document');

const {
    getDirectories,
    getCFDISasCSV,
    getPDFData,
    validateDirectory,
    checkMonthsDirectoryStatus,
    saveContactInfo,
    getContactInfo,
    monthLock
} = require('../src/controllers/directory');

const {
    getClientsOrProvidersForAUser,
    getInvitedVendorsInformation,
    resendInvitationForVendor
} = require('../src/controllers/dashboard');

const { getClientFacturas, getClientFacturasByid, uploadClientFfacturasInvoice, getFacturasFiles, deleteFacturasFiles } = require('../src/controllers/facturas');
const { getNotifications, readNotifications } = require('../src/controllers/notification');
const { getReport } = require('../src/controllers/report');

const {
    sendMessage,
    getMessage,
    deleteMessage,
    getCountUnseenMsg
} = require('../src/controllers/chat');

/* GET Status => "Health Checkup" / */

router.get('/', function (req, res) {
    if (req.user) {
        return res.send({ Success: "OK" });
    } else {
        return res.redirect('/api');
    }
});

/* USER */
router.get('/user', getUser);                                // Get User Information  
router.get('/user/:id', getUserById);                        // Get User Information
router.put('/user', updateUser);                             // Update User Information
router.delete('/user', deleteUser);                          // Delete User Information
router.post('/user/change_password', changePassword);        // User Changes Password

// Clients only routes

router.post('/user/add_vendor', addVendor);                           // Add Vendor, Option Availible only for clients
router.post('/user/add_client', addClient);                           // Add Client, Option Availible only for providers

/* DOCUMENTS */

router.get('/documents/:id', getDocumentById);                                           //  Get Documents Information  
router.post('/documents', uploadMiddleware, uploadDocument);                             //  Post Documents (Upload API)
router.post('/zip/documents', uploadMiddleware, uploadZipDocument);                    // Post Zip Documents (Zip Upload API)
router.delete('/documents/:id', deleteDocumentById);                                     //  Delete Documents
router.post('/documents/delete', deleteSelectedDocumentByIds);                // Delete Multiple Documents
router.post('/zip-extracted-documents', getZipExtractedDocuments);                    // Get Extracted Documents Information

/* DIRECTORY */
router.post('/directories/:id', getDirectories)                                      // Get Directories for a provider or a client id. (Mapping) 
router.get('/directories/:id/:year/:month/cfdis/generateCSV', getCFDISasCSV);        //  Export CSV
router.get('/directories/:id/:year/:month/generatePDF', getPDFData);        //  Export PDF
router.get('/validate_information/:id/:folder', validateDirectory);        // Validate Directory (Validation API)
router.get('/directories/:id/:year/:month/checkMonthsDirectoryStatus', checkMonthsDirectoryStatus);       // Checking the status of directories
router.post('/save/contact-info/:id', saveContactInfo);                                    // Saving client and provider contact info
router.get('/contact-info/:id', getContactInfo);                                           // getting client and provider contact info
router.post('/month-lock/:id', monthLock);
/* DASHBOARD */

router.get('/get_clients_providers_for_a_user', getClientsOrProvidersForAUser);                    // Get list of providers or Clients for a user based on what role it's on / Dashboards               
router.get('/get_invited_vendors_information', getInvitedVendorsInformation);
router.get('/resend_invitation_for_vendor', resendInvitationForVendor);

/* PAYMENTS */

// router.get('/payment');    // Get Payment Information  
// router.post('/payment');   // Create Payments
// router.put('/payment');    // Update Payment Information
// router.delete('/payment');    // Delete Payment Information

// Client Facturas
router.get('/client-facturas', getClientFacturas);
router.get('/getclientfacturas_byid/:id', getClientFacturasByid);
router.post('/upload-client-facturas-invoice', uploadMiddleware, uploadClientFfacturasInvoice);
router.get('/facturas-download/:id/:type', getFacturasFiles);
router.get('/facturas-delete-file/:id', deleteFacturasFiles);

// get Notification
router.get('/get-notification', getNotifications);
router.get('/read-notification', readNotifications);
// get report
router.post('/report', getReport);

//Chat
router.post('/send-message',sendMessage);   
router.get('/get-message/:id',getMessage);
router.get('/delete-message/:id',deleteMessage);
router.get('/unseen-msg-count/:id', getCountUnseenMsg);

module.exports = router;
