import { SpaceProps, LayoutProps, ColorProps } from 'styled-system';
/**
 * Props for Drawer component. Apart from those described below it also extends all
 * {@link SpaceProps}.
 *
 * @alias DrawerProps
 * @extends SpaceProps
 * @extends LayoutProps
 * @extends ColorProps
 * @memberof Drawer
 */
export declare type DrawerProps = SpaceProps & LayoutProps & ColorProps & {
    /** Indicates if drawer should be hidden */
    isHidden?: boolean;
    /**
     * Drawer variant
     */
    variant?: 'filter';
};
/**
 * @classdesc
 *
 * <img src="components/drawer.png" />
 *
 * Drawer component renders a huge side area where {@link BaseActionComponent} renders
 * all actions where {@link Action.showInDrawer} is set to true.
 *
 * You probably don't want to use it directly in your actions, but if you decide to set
 * `showInDrawer` to true you will probably want to use `DrawerContent` or `DrawerFooter`
 * components.
 *
 * All these components: Drawer, DrawerContent and Drawer Footer extends {@link SpaceProps}.
 *
 * ### Usage
 *
 * ```javascript
 * import { Drawer, DrawerProps, DrawerContent, DrawerFooter } from '@admin-bro/design-system'
 * ```
 *
 * @component
 * @subcategory Molecules
 * @hideconstructor
 * @see DrawerProps
 * @see {@link https://storybook.adminbro.com/?path=/story/designsystem-molecules-drawer--default Storybook}
 * @example
 * return (
 * <Box height="500px">
 *   <Drawer>
 *     <DrawerContent>
 *       <Header.H3>
 *         <Button size="icon" rounded mr="lg">
 *           <Icon icon="ChevronRight" />
 *         </Button>
 *         Edit
 *       </Header.H3>
 *       <Box my="x3" p={0}>
 *         <Button size="sm">
 *           <Icon icon="Information" />
 *           Info
 *         </Button>
 *         <Button size="sm" ml="lg">
 *           <Icon icon="Delete" />
 *           Delete
 *         </Button>
 *       </Box>
 *
 *     </DrawerContent>
 *     <DrawerFooter>
 *       <Button variant="primary">
 *         Save
 *       </Button>
 *     </DrawerFooter>
 *   </Drawer>
 * </Box>
 * )
 *
 * @section design-system
 */
declare const Drawer: import("styled-components").StyledComponent<"section", import("styled-components").DefaultTheme, DrawerProps, never>;
export { Drawer };
export default Drawer;
