const objectId = require('mongoose').Types.ObjectId;
const Mailgun = require('mailgun-js');
const Twig = require('twig');
const User = require('../models/user');
const accountsRelations = require('../models/accountsRelations');
const VendorRegistrationMap = require('../models/vendorRegistrationMap');
const {deleteInformationFromUserModel,SendRegistrationLinkViaEmail} = require('../helper');

/* Login Function. */

module.exports = {

    // Get Authenticated User information
    getUser : (req, res) => {
        // We won't reach here without authentication, this req.user should be availible.
        User.findById(req.user._id).lean().then((userData) => {

            // Check if the user Exists
            if(userData){
                // Delete Information that is not required.
                userData = deleteInformationFromUserModel(userData);

                // Return the data back
                res.status(200);
                return res.send(userData);
            }

             // if no user is find, Return this message.
             res.status(400);
             return res.send({Error: "Cannot find the requested User."});
        }).catch((err) => {
            // Console the Error
            console.log(err);

            res.status(400);
            return res.send({Error: "Cannot Process your Request."})
        })
    },

    // Get User by Id, if it belongs to authenticated user
    getUserById : async (req, res) => {

        // Fetch Authenticated User Data
        const authenticatedUser = await User.findById(req.user._id).lean();

        // Get Id for the User to be queried.
        const {id} = req.params
        if(id && objectId.isValid(id)){
            // Client & Provider Ids Data
            let clientId = "";
            let providerId = "";

            // Determine if Authenticated user a client or a provider
            if(authenticatedUser.roles == "client"){
                clientId = objectId(authenticatedUser._id);
                providerId = objectId(id);
            }else{
                clientId = objectId(id);
                providerId = objectId(authenticatedUser._id);
            }

            // Get the Account Relations Data, If it Exists then Query the Data & Send Back

            accountsRelations.findOne({clientId: clientId, providerId:providerId}).then((accountRelationData) => {
                if(!accountRelationData){
                    res.status(400);
                    return res.send({Error: "Cannot find the requested User."});
                }

                User.findOne({_id: id}).lean().then((userData) => {
                    // Check if the user Exists
                    if(userData){
                        // Delete Information that is not required.
                        userData = deleteInformationFromUserModel(userData);

                        // Return the data back
                        res.status(200);
                        return res.send(userData);
                    }

                    // if no user is find, Return this message.
                    res.status(400);
                    return res.send({Error: "Cannot find the requested User."});
                }).catch((err) => {
                    // Console the Error
                    console.log(err);

                    res.status(400);
                    return res.send({Error: "Cannot Process your Request."})
                });
            }).catch((err) => {
                // Console the Error
                console.log(err);

                res.status(400);
                return res.send({Error: "Cannot Process your Request."})
            });
        }else{
            res.status(400);
            return res.send({Error: "Cannot Process your Request."})
        }
    },

    // Update the authenticated user
    updateUser : (req, res) => {
        const {name,tel,rfc} = req.body;
        const updatedData  = {};
        name ? updatedData["name"] = name : '';
        tel ? updatedData["tel"] = tel : '';
        rfc ? updatedData["rfc"] = rfc : '';

        User.findByIdAndUpdate(req.user._id, updatedData,{new:true}).lean().then((userData) => {

            // Check if the user Exists
            if(userData){
                // Delete Information that is not required.
            userData = deleteInformationFromUserModel(userData);

            // Return the data back
            res.status(200);
            return res.send(userData);
        }

        // if no user is find, Return this message.
        res.status(400);
        return res.send({Error: "Cannot find the requested User."});

        }).catch((err) => {
            // Console the Error
            console.log(err);

            res.status(400);
            return res.send({Error: "Cannot Process your Request."})
        })

    },

    // Delete the authenticated user, Soft delete
    deleteUser : (req, res) => {
        User.findByIdAndUpdate(req.user._id,{isValid: false},{new: true}).lean().then((userData) => {

            // Check if the user Exists
            if(userData){
                // Delete Information that is not required.
                userData = deleteInformationFromUserModel(userData);

                // Return the data back
                res.status(200);
                return res.send(userData);
            }
            // if no user is find, Return this message.
            res.status(400);
            return res.send({Error: "Cannot find the requested User."});
        }).catch((err) => {
            // Console the Error
            console.log(err);

            res.status(400);
            return res.send({Error: "Cannot Process your Request."})
        })
    },

    // Change the password for authenticated user
    changePassword : async (req, res) => {

        // Get Password
        const {password} = req.body;

        // Get user information
        const user = await User.findById(req.user._id);


        // Hash Password
        const hashedPassword = await user.hashPassword(password);

        User.findByIdAndUpdate(req.user._id, {password: hashedPassword},{new:true}).lean().then((userData) => {

            // Check if the user Exists
            if(userData){
                // Delete Information that is not required.
                userData = deleteInformationFromUserModel(userData);

                // Return the data back
                res.status(200);
                return res.send(userData);
            }
            // if no user is find, Return this message.
            res.status(400);
            return res.send({Error: "Cannot find the requested User."});
        }).catch((err) => {
            // Console the Error
            console.log(err);

            res.status(400);
            return res.send({Error: "Cannot Process your Request."})
        })

    },

    addVendor : async (req,res) => {

        // Get vendor email from body
        const {email, name, rfcId} = req.body;

        // Generate username using name
        let nameArr = name.split(" ");

        let username = nameArr[0].replace(/[^a-zA-Z ]/g, "");

        if(username.length < 3 && nameArr.length > 1) username += nameArr[1].replace(/[^a-zA-Z ]/g, "");

        // Fetch Authenticated User Data
        const authenticatedUser = await User.findById(req.user._id);

        if((authenticatedUser.roles == "client" || authenticatedUser.roles == "subclient") && email) {
            const relationClientId = (authenticatedUser.roles == "subclient") ? authenticatedUser.clientId : authenticatedUser._id;

            User.findOne({email: email.toLowerCase(), username: username, roles: "provider"}).then((userToBeAdded) => {

                // Check if the email exists in the user db, if it does, assign directly & then send an email to both provider & client
                // Notifying them that they are now availbile.

                if(!userToBeAdded){

                    // If the user doesn't exist, Then Send the email to the email provided by the user, Along with referenceId appended to the url
                    // Once user clicks on that and registers, We can directly Link them up & send emails.
                    // Add the record to the vendorRegistrationMap Model, So that we can complete the registration once its done.
                    let newVendorRegistration = new VendorRegistrationMap();
                    newVendorRegistration.clientId = relationClientId;
                    newVendorRegistration.providerEmail = email;
                    newVendorRegistration.RFCId = rfcId;
                    newVendorRegistration.save().then((sucess) => {
                        if(!sucess) {
                            res.status(400);
                            return res.send({Error: "Vendor Registration Unsucessfull, please try again"});
                        }

                        SendRegistrationLinkViaEmail(email, authenticatedUser.name,authenticatedUser.email,newVendorRegistration.RFCId, 'provider')
                        res.status(200);
                        return res.send({message : 'Provider wasn"t in our records, We have send an email to the provider with the registration link.'})

                    }).catch((err) => {
                        // Console the Error
                        console.log(err);

                        res.status(400);
                        return res.send({Error: "Cannot Process your Request."})
                    });
                }else{
                    accountsRelations.findOne({clientId: relationClientId, providerId: userToBeAdded._id}).lean().then((aR) => {
                        if(aR){
                            res.status(400);
                            return res.send({message : 'Provider Already Exists.'})
                        }

                        // Add the provider-client relation to AccountsRelation Schema
                        let newAccountsRelations = new accountsRelations();
                        newAccountsRelations.clientId = relationClientId;
                        newAccountsRelations.providerId = userToBeAdded._id;
                        newAccountsRelations.save().then((sucess) => {
                            if(!sucess) {
                                res.status(400);
                                return res.send({Error: "Could not process your request"});
                            }
                            res.status(200);
                            res.send({message: "Provider Added Successfully"});

                        }).catch((err) => {
                            // Console the Error
                            console.log(err);

                            res.status(400);
                            return res.send({Error: "Cannot Process your Request."})
                        });

                    }).catch((err) => {
                        // Console the Error
                        console.log(err);

                        res.status(400);
                        return res.send({Error: "Cannot Process your Request."})
                    });
                }
            }).catch((err) => {
                // Console the Error
                console.log(err);

                res.status(400);
                return res.send({Error: "Cannot Process your Request."})
            });

        }else{
            res.status(400);
            return res.send({Error: "Providers cannot add vendors, Only Clients can."});
        }

    },

    addClient : async (req,res) => {

        // Get vendor email from body
        const {email, name, rfcId} = req.body;

         // Generate username using name
         let nameArr = name.split(" ");

         let username = nameArr[0].replace(/[^a-zA-Z ]/g, "");

         if(username.length < 3 && nameArr.length > 1) username += nameArr[1].replace(/[^a-zA-Z ]/g, "");

        // Fetch Authenticated User Data
        const authenticatedUser = await User.findById(req.user._id);

        if(authenticatedUser.roles == "provider" && email) {

            User.findOne({email: email.toLowerCase(), username: username, roles: "client"}).then((userToBeAdded) => {

                // Check if the email exists in the user db, if it does, assign directly & then send an email to both provider & client
                // Notifying them that they are now availbile.

                if(!userToBeAdded){

                    // If the user doesn't exist, Then Send the email to the email provided by the user, Along with referenceId appended to the url
                    // Once user clicks on that and registers, We can directly Link them up & send emails.
                    // Add the record to the vendorRegistrationMap Model, So that we can complete the registration once its done.
                    let newVendorRegistration = new VendorRegistrationMap();
                    newVendorRegistration.providerId = authenticatedUser._id;
                    newVendorRegistration.clientEmail = email;
                    newVendorRegistration.RFCId = rfcId;
                    newVendorRegistration.save().then((sucess) => {
                        if(!sucess) {
                            res.status(400);
                            return res.send({Error: "Client Registration Unsuccessfull, please try again"});
                        }

                        SendRegistrationLinkViaEmail(email, authenticatedUser.name,authenticatedUser.email,newVendorRegistration.RFCId, 'client')
                        res.status(200);
                        return res.send({message : 'Client wasn"t in our records, We have send an email to the client with the registration link.'})

                    }).catch((err) => {
                        // Console the Error
                        console.log(err);

                        res.status(400);
                        return res.send({Error: "Cannot Process your Request."})
                    });
                }else{
                    accountsRelations.findOne({providerId: authenticatedUser._id, clientId: userToBeAdded._id}).lean().then((aR) => {
                        if(aR){
                            res.status(400);
                            return res.send({message : 'Client Already Exists.'})
                        }

                        // Add the provider-client relation to AccountsRelation Schema
                        let newAccountsRelations = new accountsRelations();
                        newAccountsRelations.providerId = authenticatedUser._id;
                        newAccountsRelations.clientId = userToBeAdded._id;
                        newAccountsRelations.save().then((sucess) => {
                            if(!sucess) {
                                res.status(400);
                                return res.send({Error: "Could not process your request"});
                            }
                            res.status(200);
                            res.send({message: "Client Added Successfully"});

                        }).catch((err) => {
                            // Console the Error
                            console.log(err);

                            res.status(400);
                            return res.send({Error: "Cannot Process your Request."})
                        });

                    }).catch((err) => {
                        // Console the Error
                        console.log(err);

                        res.status(400);
                        return res.send({Error: "Cannot Process your Request."})
                    });
                }
            }).catch((err) => {
                // Console the Error
                console.log(err);

                res.status(400);
                return res.send({Error: "Cannot Process your Request."})
            });

        }else{
            res.status(400);
            return res.send({Error: "Clients cannot add client, Only Providers can."});
        }

    },

    sendNotificationEmail: async (req,res) => {
        try {
            const mailgun = new Mailgun({apiKey: process.env.MAIL_API, domain: process.env.MAIL_DOMAIN});
            const fromEmail = 'no-reply@kentaurus.com.mx';

            const providers = await User.find({roles: 'provider' }).lean();

            const promises = providers.map(provider => {
                let email = provider.email;
                // Get Rendered HTML
                Twig.renderFile('./templates/emails/notification.html.twig', {email:email}, (err, html) => {

                    const data = {
                        //Specify email data
                        from: fromEmail,
                        //The email to contact
                        to: email,
                        //Subject and text data
                        subject: 'Notification email, EBITSA',
                        html: html
                    }

                    // MailGun Fire Function
                    mailgun.messages().send(data, function (err, body) {
                        // If there is an error, render the error page
                        if (err) {
                            console.log("got an error on mailgun: ", err);
                            return false;
                        }
                        // Else we can greet and leave
                        else {
                            console.log(`Mail Send, To the user ${email}`);
                            return true;
                        }
                    });
                });
            });

            await Promise.all(promises);

            res.status(200);
            return res.send({Success: "Emails sent."});
        } catch (err) {
            res.status(400);
            return res.send({Error: "Something went wrong!"});
        }
    }
}