// env Load
require("dotenv").config();

// General Packages
var createError = require("http-errors");
var express = require("express");
var cors = require("cors");
var path = require("path");
var cookieParser = require("cookie-parser");
var logger = require("morgan");

// Admin Dashboard
const AdminBro = require("admin-bro");
const AdminBroExpress = require("@admin-bro/express");
const AdminBroMongoose = require("@admin-bro/mongoose");
const AdminBroExpressjs = require("admin-bro-expressjs");
const bcrypt = require("bcrypt");
const uploadFeature = require('@admin-bro/upload');
AdminBro.registerAdapter(AdminBroMongoose);

// Mongoose, MongoDB ORM
const mongoose = require("mongoose");
const User = require("./src/models/user");
const Documents = require("./src/models/documents");
const Vendors = require("./src/models/vendorRegistrationMap");
const Facturas = require("./src/models/facturas");
const clientList = require("./src/models/clientList");

const notificationModel = require('./src/models/notification');
// Resources Options
const UserOptions = require("./src/resourcesOptions/userOptions");
const DocumentOptions = require("./src/resourcesOptions/documentOptions");
const VendorOptions = require("./src/resourcesOptions/vendorRegistrationMapOptions");
const FacturasOptions = require("./src/resourcesOptions/facturasOptions");
const NotificationOptions = require("./src/resourcesOptions/notificationOptions");
// Passport Module, For authentication
const passport = require("passport");

// Authentication Module, Passport
require("./config/passport");

// Envirnomental variables
const { DB_URL } = process.env;

// Router Files
const publicRouter = require("./routes/public");
const authRouter = require("./routes/auth");
const mainRouter = require("./routes/main");

var app = express();

app.use(cors());

app.use(function (req, res, next) {
  res.header("Access-Control-Allow-Origin", "*");
  res.header(
    "Access-Control-Allow-Headers",
    "Origin, X-Requested-With, Content-Type, Accept"
  );
  next();
});

// Database connection
let connection;
const run = async () => {
  connection = await mongoose.connect(DB_URL, {
    useNewUrlParser: true,
    useUnifiedTopology: true,
  });
  console.log("DB Connected");
};

run();

// Admin
const adminBro = new AdminBro({
  resources: [
    { resource: clientList, options: {actions: {  list: { isAccessible: false,} }}},
    { resource: User, options: UserOptions.options },
    { resource: Documents, options: DocumentOptions.options },
    { resource: Vendors, options: VendorOptions.options },
    { resource: Facturas, options: FacturasOptions.options , features: [
      uploadFeature({
      provider: { 
          aws: { 
          region :'us-east-2',
          bucket: process.env.BUCKET_NAME, 
          secretAccessKey:process.env.AWS_SECRET_ACCESS_KEY
        } 
      },
      properties: {
            key: 'pdfs3path',
            bucket: 'uploadedFile.folder',
            mimeType: 'uploadedFile.type',
            size: 'uploadedFile.size',
            filename: 'uploadedFile.filename',
            file: 'PdfFile',
      },
      validation: { mimeTypes: ['application/pdf'] },
    }),
    uploadFeature({
      provider: { 
          aws: { 
          region :'us-east-2',
          bucket: process.env.BUCKET_NAME, 
          secretAccessKey:process.env.AWS_SECRET_ACCESS_KEY
        } 
      },
      properties: {
        file: `XmlFile`,
        filePath: `axmls3pathFilePath`,
        filesToDelete: `xmls3pathFilesToDelete`,
        key: `xmls3path`,
        mimeType: `xmls3pathMime`,
        bucket: `xmls3pathBucket`,
        size: `xmls3pathSize`,
            // file: `xmls3path.file`,
            // filePath: `xmls3path.file`,
            // filesToDelete: `xmls3path.filesToDelete`,
            // key: `xmls3path.key`,
            // mimeType: `xmls3path.mime`,
            // bucket: `xmls3path.bucket`,
            // size: `xmls3path.size`,
            
      },
      validation: { mimeTypes: ['text/xml'] },
    })
  ]
  },
  { resource: notificationModel, options: NotificationOptions.options },
  ],
  rootPath: "/admin",
  branding: {
    companyName: "Ebitsa Admin Panel",
    logo: "/assets/images/logo.png",
    favicon: "/assets/images/favicon.png",
  },
});
// Admin Routers
// const router = AdminBroExpress.buildRouter(adminBro);
// Build and use a router which will handle all AdminBro routes
const router = AdminBroExpressjs.buildAuthenticatedRouter(adminBro, {
  authenticate: async (email, password) => {
    const user = await User.findOne({ email });
    if (user && user.roles == 'admin') {
      const matched = await bcrypt.compare(password, user.password);
      if (matched) {
        return user;
      }
    }
    return false;
  },
  cookieName: "Ebitsa",
  cookiePassword: "session key",
});

app.use(adminBro.options.rootPath, router);

app.use(express.static("./src"));

// Express App Modules
app.use(logger("dev"));
app.use(express.json());
app.use(express.urlencoded({ extended: false }));
app.use(cookieParser());

// Public APIs, No authentication required.
app.use("/api", publicRouter);

// Authentication APIs, No authentication required.
app.use("/auth", authRouter);

// Application APIs, Authentication Required.
app.use("/", passport.authenticate("jwt", { session: false }), mainRouter);

// catch 404 and forward to error handler
app.use(function (req, res, next) {
  next(createError(404));
});

// error handler
app.use(function (err, req, res, next) {
  res.redirect("/");
});

module.exports = app;
